package com.wissamfawaz;

import java.util.Iterator;

public class DoublyLinkedList<T> implements DList<T>, Iterable<DNode<T>> {
    private DNode<T> header, trailer;
    private int size;

    public DoublyLinkedList() {
        header = new DNode<>(null, null, null);
        trailer = new DNode<>(header, null, null);
        header.setNext(trailer);
        size = 0;
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public boolean isEmpty() {
        return size == 0;
    }

    @Override
    public DNode<T> first() throws EmptyListException {
        if (isEmpty()) {
            throw new EmptyListException("List is empty!");
        }

        return header.getNext();
    }

    @Override
    public DNode<T> last() throws EmptyListException {
        if (isEmpty()) {
            throw new EmptyListException("List is empty!");
        }

        return trailer.getPrevious();
    }

    private void checkPosition(DNode<T> d) throws InvalidPositionException {
        if (d == null) {
            throw new InvalidPositionException("Null is not a valid position value!");
        }

        if (d == header || d == trailer) {
            throw new InvalidPositionException("Header/Trailer is not a valid position value!");
        }

        if (d.getPrevious() == null || d.getNext() == null) {
            throw new InvalidPositionException("Node is not part of a doubly linked list!");
        }
    }

    @Override
    public DNode<T> next(DNode<T> d) throws BoundaryViolationException, InvalidPositionException {
        checkPosition(d);

        if (d.getNext() == trailer) {
            throw new BoundaryViolationException("Cannot move past the last element!");
        }

        return d.getNext();
    }

    @Override
    public DNode<T> prev(DNode<T> d) throws BoundaryViolationException, InvalidPositionException {
        checkPosition(d);
        if (d.getPrevious() == header) {
            throw new BoundaryViolationException("Cannot move past the first element!");
        }

        return d.getPrevious();
    }

    @Override
    public T remove(DNode<T> d) throws InvalidPositionException {
        checkPosition(d);

        T toReturn = d.getElement();

        d.removeBindings();

        size--;
        return toReturn;
    }

    @Override
    public T replace(DNode<T> d, T e) throws InvalidPositionException {
        checkPosition(d);

        T toReturn = d.getElement();
        d.setElement(e);

        return toReturn;
    }

    @Override
    public DNode<T> insertFirst(T e) {
        DNode<T> newDNode = new DNode<>(header, e, header.getNext());

        header.getNext().setPrevious(newDNode);
        header.setNext(newDNode);

        size++;
        return newDNode;
    }

    @Override
    public DNode<T> insertLast(T e) {
        DNode<T> newDNode = new DNode<>(trailer.getPrevious(), e, trailer);

        trailer.getPrevious().setNext(newDNode);
        trailer.setPrevious(newDNode);

        size++;
        return newDNode;
    }

    @Override
    public DNode<T> insertBefore(DNode<T> d, T e) throws InvalidPositionException {
        checkPosition(d);

        DNode<T> newDNode = new DNode<>(d.getPrevious(), e, d);
        d.getPrevious().setNext(newDNode);
        d.setPrevious(newDNode);

        size++;
        return newDNode;
    }

    @Override
    public DNode<T> insertAfter(DNode<T> d, T e) throws InvalidPositionException {
        checkPosition(d);

        DNode<T> newDNode = new DNode<>(d, e, d.getNext());
        d.getNext().setPrevious(newDNode);
        d.setNext(newDNode);

        size++;
        return newDNode;
    }

    private class PositionsIterator implements Iterator<DNode<T>> {
        private DNode<T> current;

        public PositionsIterator() {
            try {
                current = first();
            } catch (EmptyListException e) {
                current = null;
            }
        }

        @Override
        public boolean hasNext() {
            return current != null;
        }

        @Override
        public DNode<T> next() {
            if (!hasNext()) {
                return null;
            }

            DNode<T> toReturn = current;

            try {
                if (current == last()) {
                    current = null;
                } else {
                    current = current.getNext();
                }
            } catch (EmptyListException e) {
                current = null;
            }

            return toReturn;
        }

    }

    public Iterator<DNode<T>> positions() {
        return new PositionsIterator();
    }

    @Override
    public Iterator<DNode<T>> iterator() {
        return positions();
    }
}
